{========================================================================}
{=                (c) 1995-98 SwiftSoft Ronald Dittrich                 =}
{========================================================================}
{=                          All Rights Reserved                         =}
{========================================================================}
{=  D 01099 Dresden             = Fax.: +49(0)351-8037944               =}
{=  Loewenstr.7a                = info@swiftsoft.de                     =}
{========================================================================}
{=  Actual versions on http://www.swiftsoft.de/index.html               =}
{========================================================================}
{=  This code is for reference purposes only and may not be copied or   =}
{=  distributed in any format electronic or otherwise except one copy   =}
{=  for backup purposes.                                                =}
{=                                                                      =}
{=  No Delphi Component Kit or Component individually or in a collection=}
{=  subclassed or otherwise from the code in this unit, or associated   =}
{=  .pas, .dfm, .dcu, .asm or .obj files may be sold or distributed     =}
{=  without express permission from SwiftSoft.                          =}
{=                                                                      =}
{=  For more licence informations please refer to the associated        =}
{=  HelpFile.                                                           =}
{========================================================================}
{=  $Date: 21.10.98 - 13:33:13 $                                        =}
{========================================================================}
unit Unit1;

interface

uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  Menus,
  StdCtrls,
  Grids,
  ComCtrls,
  IniFiles,
  ExtCtrls,
  MMSystem,
  MMVFW,
  MMObj,
  MMAVI,
  MMSplit,
  MMCstDlg;

type
  TMainForm = class(TForm)
    Open1: TMenuItem;
    File1: TMenuItem;
    MainMenu: TMainMenu;
    Help1: TMenuItem;
    AboutVFW1: TMenuItem;
    Edit1: TMenuItem;
    PutItem: TMenuItem;
    GetItem: TMenuItem;
    N1: TMenuItem;
    Exit1: TMenuItem;
    ListView1: TListView;
    ReopenItem: TMenuItem;
    MMSplitter1: TMMSplitter;
    no1: TMenuItem;
    N2: TMenuItem;
    AddItem: TMenuItem;
    SaveItem: TMenuItem;
    MMAVIFile1: TMMAVIFile;
    AVIMemFile: TMMAVIMemoryFile;
    Combinestreams1: TMenuItem;
    SaveasItem: TMenuItem;
    AVICompressor: TMMAVICompressor;
    TreeView1: TTreeView;
    N3: TMenuItem;
    PlayItem: TMenuItem;
    AVIOpenDialog: TMMAVIOpenDialog;
    AVISaveDialog: TMMAVISaveDialog;
    procedure AboutVFW1Click(Sender: TObject);
    procedure Open1Click(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure PutItemClick(Sender: TObject);
    procedure GetItemClick(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure TreeView1Change(Sender: TObject; Node: TTreeNode);
    procedure TreeView1DblClick(Sender: TObject);
    procedure MMAVICompressor1Save(Sender: TObject; Percent: Integer; var Abort: Boolean);
    procedure AddItemClick(Sender: TObject);
    procedure SaveItemClick(Sender: TObject);
    procedure Combinestreams1Click(Sender: TObject);
    procedure SaveasItemClick(Sender: TObject);
    procedure Edit1Click(Sender: TObject);
    procedure PlayItemClick(Sender: TObject);
  private
    FCurrFile   : TMMCustomAVIFile;

    function    Opened: Boolean;
    procedure   OpenFile(const Fn: string);
    procedure   CloseFile(New: TMMCustomAVIFile);
    procedure   UpdateEnable;
    procedure   ResizeList;
    procedure   BeginList;
    procedure   EndList;
    procedure   ClearList;
    procedure   AddDWord(const Title: string; Value: DWORD);
    procedure   AddFloat(const Title: string; Value: Extended);
    procedure   AddStr(const Title, Value: string);
    procedure   AddFlags(const Title: string; Value: DWORD; const Flags: array of DWORD; const Titles: array of string);
    procedure   AddEnum(const Title: string; Value: DWORD; const Values: array of DWORD; const Titles: array of string);
    procedure   AddWord(const Title: string; Value: WORD);
    procedure   AddRect(const Title: string; const Value: TRect);
    procedure   AddFOURCC(const Title: string; Value: DWORD);
    function    AppIni: TIniFile;
    procedure   BuildTree;
    procedure   ShowInfo(Node: TTreeNode);
    procedure   ClearTree;

    procedure   PushFn(const Fn: string);
    procedure   FillReopen;
    procedure   Reopen(const Fn: string);
    function    ReadReopen: TStrings;
    procedure   WriteReopen(List: TStrings);
    procedure   ReopenFnClick(Sender: TObject);
  end;

var
  MainForm: TMainForm;

implementation

{$R *.DFM}

uses
    MMRegs, MMWaveIO, Unit2;

{-- TMainForm -----------------------------------------------------------------}
function  TMainForm.Opened: Boolean;
begin
    Result := (FCurrFile <> nil) and FCurrFile.Opened;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.OpenFile(const Fn: string);
begin
    CloseFile(MMAVIFile1);
    MMAVIFile1.FileName := Fn;
    MMAVIFile1.OpenFile;
    FCurrFile := MMAVIFile1;
    Caption := Application.Title + ' - ' + Fn;
    UpdateEnable;
    BuildTree;
    PushFn(Fn);
    FillReopen;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AboutVFW1Click(Sender: TObject);
var
    Ver: Word;
begin
    Ver := HiWord(VideoForWindowsVersion);
    ShowMessage(Format('Video for Windows %d.%d', [Hi(Ver),Lo(Ver)]));
end;

{-- TMainForm -----------------------------------------------------------------}
function  TMainForm.AppIni: TIniFile;
begin
    Result := TIniFile.Create(ChangeFileExt(Application.ExeName,'.INI'));
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.Open1Click(Sender: TObject);
begin
    if AVIOpenDialog.Execute then
        OpenFile(AVIOpenDialog.FileName);
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.FormHide(Sender: TObject);
begin
    CloseFile(nil);
    AVIClearClipboard;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.CloseFile(New: TMMCustomAVIFile);
begin
    if Opened then
    begin
        ClearTree;
        AVICompressor.FreeStreams;
        if FCurrFile <> New then
            FCurrFile.CloseFile;
        FCurrFile := nil;
        UpdateEnable;
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.PutItemClick(Sender: TObject);
begin
    if Opened then FCurrFile.PutOnClipboard;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.Edit1Click(Sender: TObject);
begin
   GetItem.Enabled := AVIClipboardHasData;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.GetItemClick(Sender: TObject);
begin
    CloseFile(AVIMemFile);

    AVIMemFile.GetFromClipboard;
    FCurrFile := AVIMemFile;
    Caption := Application.Title + ' - Temporary file';
    UpdateEnable;
    BuildTree;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.Exit1Click(Sender: TObject);
begin
    Close;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.FormShow(Sender: TObject);
begin
    Caption := Application.Title;
    ResizeList;
    UpdateEnable;
    FillReopen;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.FormResize(Sender: TObject);
begin
    ResizeList;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.ResizeList;
begin
    ListView1.Columns[0].Width := -1;
    ListView1.Columns[1].Width := ListView1.Width - ListView1.Columns[0].Width;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.ClearList;
begin
    ListView1.Items.Clear;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddDWord(const Title: string; Value: DWORD);
begin
    AddStr(Title,Format('%8.8x (%d)',[Value,Value]));
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddFloat(const Title: string; Value: Extended);
begin
    AddStr(Title,FloatToStr(Value));
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddFlags(const Title: string; Value: DWORD; const Flags: array of DWORD; const Titles: array of string);
var
    S: string;
    i: Integer;

    procedure Add(const Part: string);
    begin
        if S = '' then
            S := Part
        else
            S := S + ' ' + Part;
    end;

begin
    for i := Low(Flags) to High(Flags) do
        if Value and Flags[i] <> 0 then
            Add(Titles[i]);
    AddStr(Title,Format('%8.8x (%s)',[Value,S]));
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddEnum(const Title: string; Value: DWORD; const Values: array of DWORD; const Titles: array of string);
var
    S: string;
    i: Integer;
begin
    S := 'unknown value';
    for i := Low(Values) to High(Values) do
        if Value = Values[i] then
        begin
            S := Titles[i];
            Break;
        end;
    AddStr(Title,Format('%8.8x (%s)',[Value,S]));
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddWord(const Title: string; Value: WORD);
begin
    AddStr(Title,Format('%4.4x (%d)',[Value,Value]));
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddRect(const Title: string; const Value: TRect);
begin
    AddStr(Title,Format('(%d,%d)-(%d,%d)',[Value.Left,Value.Top,Value.Right,Value.Bottom]));
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddFOURCC(const Title: string; Value: DWORD);
var
    S: string;
    i: Integer;
begin
    S := '';
    for i := 1 to 4 do
    begin
        S := S + Chr(Value and $FF);
        Value := Value shr 8;
    end;
    AddStr(Title,S);
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddStr(const Title, Value: string);
begin
    with ListView1.Items.Add do
    begin
        Caption := Title;
        SubItems.Add(Value);
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.BeginList;
begin
    ListView1.Items.BeginUpdate;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.EndList;
begin
    ListView1.Items.EndUpdate;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.TreeView1Change(Sender: TObject; Node: TTreeNode);
begin
    ShowInfo(Node);
    UpdateEnable;
end;


type
    TAVINode = class
    protected
        function AVIFile: TMMCustomAVIFile;
    public
        procedure ShowInfo; virtual; abstract;
    end;

    TAVIFileNode = class(TAVINode)
    public
        procedure ShowInfo; override;
    end;

    TAVIStreamNode = class(TAVINode)
    private
        FStream     : TMMAVIStream;
    protected
        property    AVIStream: TMMAVIStream read FStream;
        procedure   ShowFormatInfo; virtual;
    public
        constructor Create(AStream: TMMAVIStream);
        destructor  Destroy; override;

        procedure   Play; virtual;
        procedure   ShowInfo; override;
    end;

    TAVIAudioStreamNode = class(TAVIStreamNode)
    protected
        procedure   ShowFormatInfo; override;
    public
        procedure   Play; override;
    end;

    TAVIVideoStreamNode = class(TAVIStreamNode)
    protected
        procedure   ShowFormatInfo; override;
    public
        procedure   Play; override;
    end;

{-- TMainForm -----------------------------------------------------------------}
function  TAVINode.AVIFile: TMMCustomAVIFile;
begin
    Result := MainForm.FCurrFile;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TAVIFileNode.ShowInfo;
var
    Info: TAVIFILEINFO;
    FrameRate: Extended;
    TotalTime: Extended;
begin
    Info := AVIFile.FileInfo;
    with MainForm do
    begin
        BeginList;
        try
            ClearList;
            with Info do
            begin
                AddDWord('dwMaxBytesPerSec',dwMaxBytesPerSec);
                AddFlags('dwFlags',dwFlags,
                         [
                          AVIFILEINFO_HASINDEX,
                          AVIFILEINFO_MUSTUSEINDEX,
                          AVIFILEINFO_ISINTERLEAVED,
                          AVIFILEINFO_WASCAPTUREFILE,
                          AVIFILEINFO_COPYRIGHTED
                         ],
                         [
                          'AVIFILEINFO_HASINDEX',
                          'AVIFILEINFO_MUSTUSEINDEX',
                          'AVIFILEINFO_ISINTERLEAVED',
                          'AVIFILEINFO_WASCAPTUREFILE',
                          'AVIFILEINFO_COPYRIGHTED'
                         ]
                        );
                AddFlags('dwCaps',dwCaps,
                         [
                          AVIFILECAPS_CANREAD,
                          AVIFILECAPS_CANWRITE,
                          AVIFILECAPS_ALLKEYFRAMES,
                          AVIFILECAPS_NOCOMPRESSION
                         ],
                         [
                          'AVIFILECAPS_CANREAD',
                          'AVIFILECAPS_CANWRITE',
                          'AVIFILECAPS_ALLKEYFRAMES',
                          'AVIFILECAPS_NOCOMPRESSION'
                         ]
                        );
                AddDWord('dwStreams',dwStreams);
                AddDWord('dwSuggestedBufferSize',dwSuggestedBufferSize);
                AddDWord('dwWidth',dwWidth);
                AddDWord('dwHeight',dwHeight);
                AddDWord('dwScale',dwScale);
                AddDWord('dwRate',dwRate);
                if dwScale <> 0 then
                    FrameRate := dwRate/dwScale
                else
                    FrameRate := 0;
                AddFloat('[dwRate/dwScale (frame rate)]',FrameRate);
                AddDWord('dwLength',dwLength);
                if FrameRate = 0 then
                    TotalTime := 0
                else
                    TotalTime := dwLength/FrameRate;
                AddFloat('[dwLength/framerate) (total time)]',TotalTime);
                AddDWord('dwEditCount',dwEditCount);
                AddStr('szFileType',szFileType);
            end;
        finally
            EndList;
        end;
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
constructor TAVIStreamNode.Create(AStream: TMMAVIStream);
begin
    inherited Create;
    FStream := AStream;
    AStream.AddRef;
end;

{-- TMainForm -----------------------------------------------------------------}
destructor  TAVIStreamNode.Destroy;
begin
    if FStream <> nil then
        FStream.Release;
    inherited Destroy;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure   TAVIStreamNode.ShowInfo;
var
    FrameRate   : Extended;
    TotalTime   : Extended;
begin
    with MainForm do
    begin
        BeginList;
        try
            ClearList;
            with AVIStream.StreamInfo do
            begin
                AddFOURCC('fccType',fccType);
                AddFOURCC('fccHandler',fccHandler);

                AddFlags('dwFlags',dwFlags,
                         [AVISTREAMINFO_DISABLED,AVISTREAMINFO_FORMATCHANGES],
                         ['AVISTREAMINFO_DISABLED','AVISTREAMINFO_FORMATCHANGES']);
                AddDWord('dwCaps',dwCaps);

                AddWord('wPriority',wPriority);
                AddWord('wLanguage',wLanguage);

                AddDWord('dwScale',dwScale);
                AddDWord('dwRate',dwRate);
                if dwScale = 0 then
                    FrameRate := 0
                else
                    FrameRate := dwRate/dwScale;
                AddFloat('[dwRate/dwScale (frame rate)]',FrameRate);
                AddDWord('dwStart',dwStart);
                AddDWord('dwLength',dwLength);
                if FrameRate = 0 then
                    TotalTime := 0
                else
                    TotalTime := dwLength/FrameRate;
                AddFloat('[dwLength/framerate) (total time)]',TotalTime);
                AddDWord('dwInitialFrames',dwInitialFrames);
                AddDWord('dwSuggestedBufferSize',dwSuggestedBufferSize);
                AddDWord('dwQuality',dwQuality);
                AddDWord('dwSampleSize',dwSampleSize);

                AddRect('rcFrame',rcFrame);

                AddDWord('dwEditCount',dwEditCount);
                AddDWord('dwFormatChangeCount',dwFormatChangeCount);

                AddStr('szName',szName);
            end;
            AddStr('-----','');
            ShowFormatInfo;
        finally
            EndList;
        end;
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure   TAVIStreamNode.ShowFormatInfo;
begin
    MainForm.AddStr('format','not available');
end;

{-- TMainForm -----------------------------------------------------------------}
procedure   TAVIStreamNode.Play;
begin
end;

{-- TMainForm -----------------------------------------------------------------}
procedure   TAVIAudioStreamNode.ShowFormatInfo;
var
    pwfx: PWaveFormatEx;
    S: string;
begin
    with TMMAVIAudioWorker.Create(AVIStream) do
        try
            pwfx := Format;
            with MainForm do
            begin
                AddStr('format','audio');
                wioGetFormatName(pwfx,S);
                AddStr('name',S);
                wioGetFormat(pwfx,S);
                AddStr('details',S);
            end;
        finally
            Free;
        end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure   TAVIAudioStreamNode.Play;
begin
    PlayStream(AVIStream);
end;

{-- TMainForm -----------------------------------------------------------------}
procedure   TAVIVideoStreamNode.ShowFormatInfo;
var
    pbi: PBitmapInfo;
begin
    with TMMAVIVideoWorker.Create(AVIStream) do
        try
            pbi := Format[AVIStream.StartPos];
            with MainForm, pbi.bmiHeader do
            begin
                AddStr('format','video');
                AddDWord('biSize',biSize);
                AddDWord('biWidth',biWidth);
                AddDWord('biHeight',biHeight);
                AddWord('biPlanes',biPlanes);
                AddWord('biBitCount',biBitCount);
                if biCompression in [BI_RGB,BI_RLE8,BI_RLE4,BI_BITFIELDS] then
                    AddEnum('biCompression',biCompression,
                            [BI_RGB,BI_RLE8,BI_RLE4,BI_BITFIELDS],
                            ['BI_RGB','BI_RLE8','BI_RLE4','BI_BITFIELDS'])
                else
                    AddFOURCC('biCompression',biCompression);
                AddDWord('biSizeImage',biSizeImage);
                AddDWord('biXPelsPerMeter',biXPelsPerMeter);
                AddDWord('biYPelsPerMeter',biYPelsPerMeter);
                AddDWord('biClrUsed',biClrUsed);
                AddDWord('biClrImportant',biClrImportant);
            end;
        finally
            Free;
        end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure   TAVIVideoStreamNode.Play;
begin
    PlayStream(AVIStream);
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.BuildTree;
var
    Title: string;

    procedure   AddStreams(Root: TTreeNode);
    var
        i: Integer;
    begin
        with FCurrFile do
        for i := 0 to StreamCount - 1 do
            with TreeView1.Items.AddChild(Root,Streams[i].Name) do
                 case Streams[i].StreamType of
                        stVideo: Data := TAVIVideoStreamNode.Create(Streams[i]);
                        stAudio: Data := TAVIAudioStreamNode.Create(Streams[i]);
                           else  Data := TAVIStreamNode.Create(Streams[i]);
                 end;
    end;

begin
    if not Opened then
        Exit;

    with TreeView1.Items do
    begin
        BeginUpdate;
        try
            ClearTree;
            try
                if FCurrFile = MMAVIFile1 then
                    Title := MMAVIFile1.FileName
                else
                    Title := 'Temporary file';
                with Add(nil,Title) do
                begin
                    Data := TAVIFileNode.Create;
                end;
                AddStreams(TreeView1.Items[0]);
                TreeView1.Items[0].Expand(True);
                TreeView1.Selected := TreeView1.Items[0];
                TreeView1.SetFocus;
            except
                ClearTree;
                raise;
            end;
        finally
            EndUpdate;
        end;
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.ShowInfo(Node: TTreeNode);
begin
    if Node.Data <> nil then
        TAVINode(Node.Data).ShowInfo;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.ClearTree;
var
    i: Integer;
begin
    with TreeView1 do
        for i := 0 to Items.Count - 1 do
            if Items[i].Data <> nil then
            begin
                TObject(Items[i].Data).Free;
                Items[i].Data := nil;
            end;
    TreeView1.Items.Clear;
end;

const
    MaxReopen = 24;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.PushFn(const Fn: string);
var
    List: TStrings;
    Ind : Integer;
begin
    List := ReadReopen;
    try
        Ind := List.IndexOf(Fn);
        if Ind <> -1 then
            List.Delete(Ind);
        if List.Count >= MaxReopen then
            List.Delete(List.Count-1);
        List.Insert(0,Fn);
        WriteReopen(List);
    finally
        List.Free;
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.FillReopen;
var
    List: TStrings;
    i   : Integer;

    procedure   AddEmpty;
    var
        Item: TMenuItem;
    begin
        Item := TMenuItem.Create(Self);
        with Item do
        begin
            Enabled := False;
            Caption := '(no file)';
        end;
        ReopenItem.Add(Item);
    end;

    procedure   AddItem(const Fn: string);
    var
        Item: TMenuItem;
    begin
        Item := TMenuItem.Create(Self);
        with Item do
        begin
            Caption := Fn;
            OnClick := ReopenFnClick;
        end;
        ReopenItem.Add(Item);
    end;

begin
    with ReopenItem do
    begin
        while Count > 0 do
            Items[0].Free;
    end;
    List := ReadReopen;
    try
        if List.Count = 0 then
            AddEmpty
        else
            for i := 0 to List.Count - 1 do
                AddItem(List[i]);
    finally
        List.Free;
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.Reopen(const Fn: string);
begin
    OpenFile(Fn);
end;

{-- TMainForm -----------------------------------------------------------------}
function  TMainForm.ReadReopen: TStrings;
var
    i, N: Integer;
begin
    Result := TStringList.Create;
    try
        with AppIni do
            try
                N := ReadInteger('Reopen','Count',0);
                for i := 1 to N do
                    Result.Add(ReadString('Reopen','Item'+IntToStr(i),''));
            finally
                Free;
            end;
    except
        Result.Free;
        raise;
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.WriteReopen(List: TStrings);
var
    i: Integer;
begin
    with AppIni do
        try
            WriteInteger('Reopen','Count',List.Count);
            for i := 0 to List.Count - 1 do
                WriteString('Reopen','Item'+IntToStr(i+1),List[i]);
        finally
            Free;
        end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.ReopenFnClick(Sender: TObject);
begin
    with Sender as TMenuItem do
        Reopen(Caption);
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.TreeView1DblClick(Sender: TObject);
begin
    with TreeView1 do
    if Selected <> nil then
       if Selected.Data <> nil then
          if TObject(Selected.Data) is TAVIStreamNode then
             (TObject(Selected.Data) as TAVIStreamNode).Play;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.MMAVICompressor1Save(Sender: TObject; Percent: Integer; var Abort: Boolean);
begin
    Caption := IntToStr(Percent)+'%';
    if GetAsyncKeyState(VK_ESCAPE) < 0 then Abort := True;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.AddItemClick(Sender: TObject);
var
    mms: TMMAVIStream;
begin
    if (TreeView1.Selected <> nil) and (TObject(TreeView1.Selected.Data) is TAVIStreamNode) then
    begin
        mms := TMMAVIStream.Create;
        try
            mms := TAVIStreamNode(TreeView1.Selected.Data).FStream;
            AVICompressor.AddStream(mms);
        finally
            mms.Release;
        end;
    end;
    UpdateEnable;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.SaveItemClick(Sender: TObject);
begin
    if AVISaveDialog.Execute then
       if AVICompressor.ChooseOptions then
          AVICompressor.SaveToFile(AVISaveDialog.FileName);
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.UpdateEnable;
begin
    PlayItem.Enabled := Opened;
    PutItem.Enabled := Opened;
    if not Opened then ClearList;
    AddItem.Enabled := (TreeView1.Selected <> nil) and (TObject(TreeView1.Selected.Data) is TAVIStreamNode);
    SaveItem.Enabled := AVICompressor.StreamCount > 0;
    SaveAsItem.Enabled := Opened and (FCurrFile.StreamCount > 0);
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.Combinestreams1Click(Sender: TObject);
var
    List: TList;
    i: Integer;
begin
    List := TList.Create;
    try
        for i := 0 to AVICompressor.StreamCount - 1 do
            List.Add(AVICompressor.Streams[i]);
        AVIMemFile.CombineStreams(List);
        CloseFile(AVIMemFile);
        FCurrFile := AVIMemFile;
        Caption := 'Temporary file';
        UpdateEnable;
        BuildTree;
    finally
        List.Free;
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.SaveAsItemClick(Sender: TObject);
begin

   if (FCurrFile <> nil) and (FCurrFile.TypedStreams[stVideo,0].LengthTime > 5*60000) then
   begin
      MessageDlg('This demo is limited so this file is to large to save',mtInformation,[mbOK],0);
      exit;
   end;

    if AVISaveDialog.Execute then
    begin
        AVICompressor.FreeStreams;
        AVICompressor.AddFile(FCurrFile);
        if AVICompressor.ChooseOptions then
           AVICompressor.SaveToFile(AVISaveDialog.FileName);
    end;
end;

{-- TMainForm -----------------------------------------------------------------}
procedure TMainForm.PlayItemClick(Sender: TObject);
begin
   PlayFile(FCurrFile);
end;

end.
